// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (c) 2019 Linaro Limited
 * Author: AKASHI Takahiro
 *
 * Unit test for rsa_verify() function
 */

#include <command.h>
#include <image.h>
#include <test/lib.h>
#include <test/test.h>
#include <test/ut.h>
#include <u-boot/rsa.h>

#ifdef CONFIG_RSA_VERIFY_WITH_PKEY
/*
 * openssl genrsa 2048 -out private.pem
 * openssl rsa -in private.pem -pubout -outform der -out public.der
 * dd if=public.der of=public.raw bs=24 skip=1
 */
static unsigned char public_key[] = {
	0x30, 0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0xca, 0x25, 0x23,
	0xe0, 0x0a, 0x4d, 0x8f, 0x56, 0xfc, 0xc9, 0x06, 0x4c, 0xcc, 0x94, 0x43,
	0xe0, 0x56, 0x44, 0x6e, 0x37, 0x54, 0x87, 0x12, 0x84, 0xf9, 0x07, 0x4f,
	0xe4, 0x23, 0x40, 0xc3, 0x43, 0x84, 0x37, 0x86, 0xd3, 0x9d, 0x95, 0x1c,
	0xe4, 0x8a, 0x66, 0x02, 0x09, 0xe2, 0x3d, 0xce, 0x2c, 0xc6, 0x02, 0x6a,
	0xd4, 0x65, 0x61, 0xff, 0x85, 0x6f, 0x88, 0x63, 0xba, 0x31, 0x62, 0x1e,
	0xb7, 0x95, 0xe9, 0x08, 0x3c, 0xe9, 0x35, 0xde, 0xfd, 0x65, 0x92, 0xb8,
	0x9e, 0x71, 0xa4, 0xcd, 0x47, 0xfd, 0x04, 0x26, 0xb9, 0x78, 0xbf, 0x05,
	0x0d, 0xfc, 0x00, 0x84, 0x08, 0xfc, 0xc4, 0x4b, 0xea, 0xf5, 0x97, 0x68,
	0x0d, 0x97, 0xd7, 0xff, 0x4f, 0x92, 0x82, 0xd7, 0xbb, 0xef, 0xb7, 0x67,
	0x8e, 0x72, 0x54, 0xe8, 0xc5, 0x9e, 0xfd, 0xd8, 0x38, 0xe9, 0xbe, 0x19,
	0x37, 0x5b, 0x36, 0x8b, 0xbf, 0x49, 0xa1, 0x59, 0x3a, 0x9d, 0xad, 0x92,
	0x08, 0x0b, 0xe3, 0xa4, 0xa4, 0x7d, 0xd3, 0x70, 0xc0, 0xb8, 0xfb, 0xc7,
	0xda, 0xd3, 0x19, 0x86, 0x37, 0x9a, 0xcd, 0xab, 0x30, 0x96, 0xab, 0xa4,
	0xa2, 0x31, 0xa0, 0x38, 0xfb, 0xbf, 0x85, 0xd3, 0x24, 0x39, 0xed, 0xbf,
	0xe1, 0x31, 0xed, 0x6c, 0x39, 0xc1, 0xe5, 0x05, 0x2e, 0x12, 0x30, 0x36,
	0x73, 0x5d, 0x62, 0xf3, 0x82, 0xaf, 0x38, 0xc8, 0xca, 0xfa, 0xa1, 0x99,
	0x57, 0x3c, 0xe1, 0xc1, 0x7b, 0x05, 0x0b, 0xcc, 0x2e, 0xa9, 0x10, 0xc8,
	0x68, 0xbd, 0x27, 0xb6, 0x19, 0x9c, 0xd2, 0xad, 0xb3, 0x1f, 0xca, 0x35,
	0x6e, 0x84, 0x23, 0xa1, 0xe9, 0xa4, 0x4c, 0xab, 0x19, 0x09, 0x79, 0x6e,
	0x3c, 0x7b, 0x74, 0xfc, 0x33, 0x05, 0xcf, 0xa4, 0x2e, 0xeb, 0x55, 0x60,
	0x05, 0xc7, 0xcf, 0x3f, 0x92, 0xac, 0x2d, 0x69, 0x0b, 0x19, 0x16, 0x79,
	0x75, 0x02, 0x03, 0x01, 0x00, 0x01
};

static unsigned int public_key_len = 270;

/*
 * dd if=/dev/urandom of=data.raw bs=512 count=1
 */
static unsigned char data_raw[] = {
	0x3e, 0x48, 0x6e, 0xef, 0x83, 0xd1, 0x4c, 0xfd, 0x92, 0x47, 0x92, 0xd7,
	0xf6, 0x16, 0x25, 0x0a, 0xdf, 0xe2, 0xb6, 0x6c, 0xe7, 0xe0, 0x55, 0xb2,
	0x70, 0x66, 0xf0, 0xe5, 0xdc, 0xaf, 0xd3, 0x2e, 0xc1, 0x3e, 0x5c, 0x4b,
	0xb5, 0xa7, 0x23, 0x1f, 0x2c, 0xce, 0xf8, 0x83, 0x00, 0x6d, 0xeb, 0xdd,
	0x19, 0x71, 0x13, 0xb4, 0xae, 0x5c, 0xa8, 0xae, 0x52, 0xc8, 0xe1, 0x77,
	0x9e, 0x98, 0x75, 0xbc, 0xef, 0x36, 0x9f, 0x0c, 0x14, 0xed, 0x1a, 0x0a,
	0x4f, 0x6c, 0xa4, 0xb1, 0xbb, 0x0e, 0x43, 0x93, 0x12, 0xfc, 0x2e, 0x82,
	0x93, 0x4e, 0xcb, 0xa2, 0xcd, 0x59, 0x3f, 0xc5, 0x11, 0x38, 0x3a, 0x88,
	0xc3, 0xcf, 0xf9, 0x61, 0xa8, 0x9e, 0x96, 0xb6, 0xbf, 0xa6, 0x5b, 0x0d,
	0xd9, 0xbd, 0x05, 0x4c, 0xbe, 0xed, 0x86, 0xca, 0x10, 0x63, 0x72, 0x75,
	0x4b, 0xbd, 0x86, 0x42, 0x30, 0x9d, 0x54, 0x4e, 0x12, 0xda, 0xf4, 0xb4,
	0xfd, 0xd9, 0x54, 0x95, 0x8f, 0x83, 0xc2, 0x63, 0x44, 0xdd, 0x96, 0x1a,
	0xd0, 0x7c, 0xcf, 0xcb, 0x16, 0xd6, 0xff, 0xa3, 0xbb, 0xeb, 0x24, 0x06,
	0xbf, 0x81, 0xd0, 0x29, 0x76, 0x19, 0x66, 0x84, 0xfc, 0x49, 0xde, 0x7b,
	0x5d, 0xd2, 0x27, 0x58, 0x21, 0x7b, 0xff, 0x4d, 0x64, 0xf3, 0x89, 0xe3,
	0xea, 0xb6, 0x54, 0x4e, 0xb1, 0x62, 0x52, 0x89, 0xe3, 0x22, 0xf2, 0x26,
	0x3e, 0x4f, 0x43, 0x58, 0x78, 0x91, 0x55, 0xbc, 0x1e, 0xd6, 0x97, 0xfc,
	0x0b, 0x85, 0x4c, 0x92, 0x9c, 0xbf, 0xc4, 0xb1, 0x62, 0x93, 0x27, 0xa9,
	0xb2, 0xf4, 0xb4, 0x7a, 0xfb, 0x56, 0xe5, 0x8f, 0xe1, 0x94, 0x4d, 0xfd,
	0xe4, 0x72, 0x8d, 0xa9, 0x71, 0x65, 0xcb, 0x2e, 0x6d, 0x39, 0xd5, 0x95,
	0xe7, 0x3f, 0xab, 0xaa, 0x7a, 0x74, 0x84, 0x25, 0x4b, 0x42, 0x1e, 0xd3,
	0x86, 0xca, 0x47, 0x4a, 0xf0, 0x24, 0x81, 0x24, 0xb0, 0xe1, 0xbb, 0x6c,
	0x3f, 0x2a, 0xa0, 0xb8, 0xeb, 0xd6, 0x01, 0xce, 0x63, 0x51, 0xe1, 0x81,
	0xd2, 0x32, 0x43, 0x56, 0x44, 0x4a, 0x6b, 0x51, 0x24, 0xa2, 0xc7, 0x39,
	0x7c, 0x54, 0xda, 0xf8, 0xd4, 0x93, 0x7c, 0x8e, 0x4e, 0x9d, 0x15, 0x08,
	0xce, 0x27, 0xd8, 0x28, 0xb0, 0x5b, 0x75, 0x32, 0x43, 0xe8, 0xd6, 0xbf,
	0x12, 0xd5, 0xc5, 0x12, 0x8e, 0xeb, 0x77, 0x8f, 0x00, 0xde, 0x45, 0x1e,
	0xdd, 0xf3, 0xef, 0x43, 0x99, 0x79, 0x86, 0xea, 0x01, 0xce, 0xf2, 0x4d,
	0xa0, 0xfe, 0x5a, 0x55, 0xc0, 0x1f, 0xce, 0xe8, 0xbe, 0xc2, 0x66, 0xdb,
	0xcb, 0x3f, 0xa5, 0x48, 0xa1, 0xe2, 0x49, 0xa1, 0x29, 0x65, 0x5b, 0x62,
	0x39, 0xcc, 0xef, 0xbe, 0x86, 0xb7, 0xe3, 0x44, 0x67, 0x04, 0x04, 0xb1,
	0xec, 0xd8, 0xb2, 0xb2, 0x38, 0xbc, 0x10, 0xea, 0x7a, 0x0e, 0xa4, 0xa4,
	0xcb, 0x21, 0xd9, 0xc7, 0xb4, 0x0b, 0xb8, 0x39, 0xb4, 0x07, 0x53, 0x3f,
	0xb9, 0x55, 0x55, 0xa1, 0x6f, 0x11, 0x49, 0xc0, 0x94, 0x77, 0xaf, 0x76,
	0x97, 0x7f, 0x31, 0x08, 0xdd, 0x72, 0x48, 0x72, 0xf8, 0x11, 0x4f, 0x69,
	0x10, 0xef, 0x23, 0x06, 0xf3, 0x34, 0xac, 0xee, 0x97, 0x89, 0x41, 0x1c,
	0x36, 0x38, 0xb1, 0x80, 0x96, 0x7a, 0x9e, 0x72, 0xab, 0x25, 0xeb, 0xce,
	0x7b, 0xb8, 0x5d, 0xc8, 0xef, 0xa4, 0x73, 0xa1, 0xa6, 0x8f, 0x01, 0x54,
	0xce, 0x58, 0x19, 0xe5, 0x7e, 0xfa, 0x77, 0x08, 0x9d, 0x53, 0xc1, 0xcc,
	0x08, 0xe8, 0x1d, 0xe0, 0x82, 0x5e, 0xe1, 0xe6, 0xbd, 0xbb, 0x59, 0x7e,
	0x12, 0x9c, 0x39, 0x60, 0x23, 0xf7, 0xbe, 0x0a, 0x7c, 0x48, 0x12, 0xa0,
	0x84, 0x04, 0x3f, 0xa1, 0x6e, 0x92, 0xcd, 0xa0, 0xac, 0xee, 0x0b, 0xbc,
	0x18, 0x30, 0x28, 0xbd, 0xf5, 0xfa, 0x3a, 0x35
};

static unsigned int data_raw_len = 512;

/*
 * openssl dgst -sha256 -sign private.key -out data.enc data.raw
 */
unsigned char data_enc[] = {
	0xa7, 0x4a, 0x12, 0x8f, 0xee, 0x65, 0x4b, 0xcd, 0x88, 0xca, 0x4d, 0xed,
	0xe3, 0x04, 0xe7, 0x7c, 0x59, 0xbf, 0x2f, 0xad, 0x95, 0x73, 0x5b, 0x2c,
	0x4e, 0xb5, 0xda, 0x5e, 0x3a, 0x6d, 0xb4, 0xc5, 0x84, 0x0c, 0xd2, 0x4a,
	0x62, 0x0d, 0x5f, 0xba, 0x10, 0xee, 0xb1, 0x2a, 0xe1, 0xfe, 0x50, 0x18,
	0x97, 0xcc, 0xea, 0x26, 0x62, 0x33, 0x5a, 0x1d, 0x51, 0x38, 0x52, 0x89,
	0x4d, 0xa7, 0x18, 0xff, 0xa6, 0xc8, 0xd4, 0x7a, 0xc0, 0xa6, 0x22, 0xdf,
	0x41, 0x89, 0x93, 0x9b, 0xe7, 0x9e, 0xc1, 0xc8, 0x80, 0xda, 0x1a, 0x3f,
	0xa4, 0x7a, 0xd0, 0x07, 0xcb, 0x5c, 0xa4, 0x75, 0x12, 0x54, 0x78, 0x67,
	0xbf, 0xe6, 0xae, 0x1e, 0x56, 0x33, 0x9e, 0xe0, 0x6e, 0x33, 0xa7, 0x58,
	0xb0, 0x47, 0x49, 0xa8, 0x37, 0xdb, 0x82, 0x4b, 0xbd, 0x32, 0x9c, 0xdc,
	0xf4, 0x67, 0x17, 0x24, 0x55, 0xfd, 0x83, 0x1e, 0xc8, 0xb4, 0x5c, 0xf9,
	0x15, 0x6c, 0x5e, 0xaa, 0x72, 0x03, 0x9e, 0x7c, 0x17, 0xf5, 0x7c, 0x37,
	0x96, 0x00, 0xb0, 0xd8, 0xaa, 0x05, 0xfa, 0xaa, 0xa1, 0x78, 0x77, 0xd5,
	0x09, 0xdd, 0x05, 0xc7, 0xe2, 0x9f, 0x68, 0xc7, 0xf8, 0xfb, 0x0b, 0x6f,
	0x18, 0x1e, 0xcc, 0x93, 0xd3, 0x3f, 0xc9, 0x26, 0x29, 0x64, 0xe7, 0x15,
	0xdc, 0xb8, 0x19, 0x10, 0x24, 0x55, 0x55, 0x3b, 0x79, 0xa1, 0x65, 0x12,
	0xe0, 0x0b, 0x88, 0x44, 0x4c, 0xea, 0x85, 0x5a, 0x6b, 0x2d, 0x45, 0x6e,
	0xe7, 0x83, 0x6f, 0x3a, 0xaa, 0x1e, 0xf1, 0x9c, 0x8f, 0xdc, 0xb9, 0x37,
	0xa2, 0x15, 0x61, 0x93, 0x06, 0x23, 0xf5, 0xfe, 0xf0, 0xf8, 0x2b, 0xf7,
	0xc0, 0x68, 0x67, 0xf6, 0x4e, 0x08, 0x0d, 0x0d, 0x08, 0xbe, 0xfb, 0x2c,
	0x4c, 0xe7, 0xd7, 0x1a, 0xad, 0xd9, 0x98, 0xa1, 0x8d, 0x94, 0x1c, 0xd1,
	0x89, 0x06, 0xc9, 0x3a
};

static unsigned int data_enc_len = 256;

/**
 * lib_rsa_verify_valid() - unit test for rsa_verify()
 *
 * Test rsa_verify() with valid hash
 *
 * @uts:	unit test state
 * Return:	0 = success, 1 = failure
 */
static int lib_rsa_verify_valid(struct unit_test_state *uts)
{
	struct image_sign_info info;
	struct image_region reg;
	int ret;

	memset(&info, '\0', sizeof(info));
	info.name = "sha256,rsa2048";
	info.padding = image_get_padding_algo("pkcs-1.5");
	info.checksum = image_get_checksum_algo("sha256,rsa2048");
	info.crypto = image_get_crypto_algo(info.name);

	info.key = public_key;
	info.keylen = public_key_len;

	reg.data = data_raw;
	reg.size = data_raw_len;
	ret = rsa_verify(&info, &reg, 1, data_enc, data_enc_len);
	ut_assertf(ret == 0, "verification unexpectedly failed (%d)\n", ret);

	return CMD_RET_SUCCESS;
}

LIB_TEST(lib_rsa_verify_valid, 0);

/**
 * lib_rsa_verify_invalid() - unit test for rsa_verify()
 *
 * Test rsa_verify() with invalid hash
 *
 * @uts:	unit test state
 * Return:	0 = success, 1 = failure
 */
static int lib_rsa_verify_invalid(struct unit_test_state *uts)
{
	struct image_sign_info info;
	struct image_region reg;
	unsigned char ctmp;
	int ret;

	memset(&info, '\0', sizeof(info));
	info.name = "sha256,rsa2048";
	info.padding = image_get_padding_algo("pkcs-1.5");
	info.checksum = image_get_checksum_algo("sha256,rsa2048");
	info.crypto = image_get_crypto_algo(info.name);

	info.key = public_key;
	info.keylen = public_key_len;

	/* randomly corrupt enc'ed data */
	ctmp = data_enc[data_enc_len - 10];
	data_enc[data_enc_len - 10] = 0x12;

	reg.data = data_raw;
	reg.size = data_raw_len;
	ret = rsa_verify(&info, &reg, 1, data_enc, data_enc_len);

	/* revert a change */
	data_enc[data_enc_len - 10] = ctmp;

	ut_assertf(ret != 0, "verification unexpectedly succeeded\n");

	return CMD_RET_SUCCESS;
}

LIB_TEST(lib_rsa_verify_invalid, 0);
#endif /* RSA_VERIFY_WITH_PKEY */
